<?php
include '../../config.php';
date_default_timezone_set('Africa/Dar_es_Salaam');

// Handle CORS
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization");

// Handle preflight request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

try {
    // Decode JSON input
    $data = json_decode(file_get_contents("php://input"), true);

    // Validate input
    $buyer_id = $data['buyer_id'] ?? 1; // Default to buyer_id 1 if not provided
    $total_amount = $data['total_amount'] ?? null;
    $order_items = $data['order_items'] ?? [];
    $status = isset($data['status']) ? $data['status'] : 0;
    $shop_id = $data['shop_id'] ?? null;  // Optional shop_id
    $order_date = date('Y-m-d H:i:s');

    if (is_null($total_amount) || empty($order_items)) {
        echo json_encode(['success' => false, 'message' => 'Invalid input']);
        http_response_code(400); // Bad request
        exit;
    }

    // Insert order (imehaririwa ili kujumuisha shop_id)
    $query = "INSERT INTO orders (buyer_id, total_amount, order_date, status, shop_id) VALUES (?, ?, ?, ?, ?)";
    $stmt = $conn->prepare($query);
    // Bind parameters: buyer_id (int), total_amount (double), status (int), shop_id (int)
    // Ikiwa shop_id ni null, itakuwa null katika database (badilisha table column ili iwe nullable)
    $stmt->bind_param("idsii", $buyer_id, $total_amount, $order_date, $status, $shop_id);
    if (!$stmt->execute()) {
        throw new Exception('Failed to create order: ' . $stmt->error);
    }

    $order_id = $stmt->insert_id;

    // Insert order items and update product quantity & orders count
    foreach ($order_items as $item) {
        $product_id = $item['product_id'];
        $quantity = $item['quantity']; // Inakuja kama decimal string
        $price = $item['price'];       // Inakuja kama decimal string
        

        // Convert to appropriate types (double) 
        $quantity = (double)$quantity;
        $price = (double)$price;

        // Insert into order_items
        $query = "INSERT INTO order_items (order_id, product_id, quantity, price) VALUES (?, ?, ?, ?)";
        $stmt = $conn->prepare($query);
        $stmt->bind_param("iidd", $order_id, $product_id, $quantity, $price);
        if (!$stmt->execute()) {
            throw new Exception('Failed to add order items: ' . $stmt->error);
        }

        // Update product quantity & orders count
        $query = "UPDATE product SET quantity = quantity - ?, orders = orders + ? WHERE id = ? AND quantity >= ?";
        $stmt = $conn->prepare($query);
        $stmt->bind_param("ddid", $quantity, $quantity, $product_id, $quantity);
        if (!$stmt->execute()) {
            throw new Exception('Failed to update product quantity and orders: ' . $stmt->error);
        }
        
        // Check if update was successful
        if ($stmt->affected_rows === 0) {
            throw new Exception('Insufficient stock for product ID ' . $product_id);
        }
    }

    echo json_encode(['success' => true, 'message' => 'Order processed successfully']);
} catch (Exception $e) {
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    http_response_code(500); // Internal server error
}
?>
